/*

oersync - a utility for syncing oer+MySQL admins/users

Copyright (C) 2002 EQU <equ@equnet.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

*/

#include "oersync.h"

int loadconf(char *);
int parseconf(char *);
int dbconnect(struct mysqldb *);
int emptyline(char *);
int iscomment(char *);
int parse(char *, int, char *, char *, int, int);
int oer_doquery(struct mysqldb *, char *, const char *, ...);
int syncadmins(void);
int syncusers(char *);
int process(void);
void oer_debug(int, const char *, ...);
void striplf(char *);
static void handlesignal(int);

int loopforever;
struct mysqldb mysqldbfrom;
struct mysqldb mysqldbto;
struct mysqldb mysqldbtemp;
char identfrom[IDENTLEN + 1];
char identto[IDENTLEN + 1];
char identtemp[IDENTLEN + 1];
int interval;
int do_debug;
int do_console;
int do_nothing;
int do_output;
int do_tempoutput;
/* synctypes */
int do_admins;
int do_autheds;
int do_hostmasks;
int do_passwords;
int do_timestamps;
int do_users;
int do_twoway;
int do_singlerun;
FILE *logto;

int main(int argc, char *argv[])
{
	int c;
	int do_help;
	int do_version;
	int retval;
	pid_t newpid;
	time_t now;
        char config[TINYSTRINGLEN + 1];
        char stringbuffer[HUGESTRINGLEN + 1];
        struct timeval tv;
        struct timeval tv1;
        struct timeval tv2;
#ifdef NEW_SIGNALS
        struct sigaction act;
        struct sigaction old;
#endif
        extern char *optarg;

	do_help = 0;
	do_version = 0;
	do_debug = DEBUG_INFO;
	do_console = 0;
	do_nothing = 0;
	do_output = 0;
	do_tempoutput = 1;
	do_admins = 0;
	do_autheds = 0;
	do_hostmasks = 0;
	do_passwords = 0;
	do_timestamps = 0;
	do_users = 0;
	do_twoway = 0;
	do_singlerun = 0;
        strncpy(config, "oersync.conf", TINYSTRINGLEN);
	while((c = getopt(argc, argv, "12cd:f:hnov")) > 0) {
                switch(c) {
		case '1':
                        do_singlerun = 1;
                        break;
		case '2':
                        do_twoway = 1;
                        break;
		case 'c':
                        do_console = 1;
                        break;
		case 'd':
                        do_debug = atoi(optarg);
                        break;
		case 'f':
                        strncpy(config, optarg, TINYSTRINGLEN);
                        break;
		case 'h':
                        do_help = 1;
                        break;
		case 'n':
                        do_nothing = 1;
                        break;
		case 'o':
                        do_output = 1;
                        break;
		case 'v':
                        do_version = 1;
                        break;
		default:
                        exit(EXIT_FAILURE);
                }
        }
	do_tempoutput = 1;
        if(do_help) {
                oer_debug(DEBUG_INFO, "usage: %s [-c] [-d <debug level>] [-f <config>] [-h] [-n] [-o] [-v]\n", argv[0]);
        }
        if(do_version) {
                oer_debug(DEBUG_INFO, "%s\n", OERSYNC_VERSION);
        }
	if(do_help || do_version) {
                exit(EXIT_SUCCESS);
	}
#ifdef NEW_SIGNALS
        sigemptyset(&act.sa_mask);
        act.sa_flags = SA_RESTART;
        act.sa_handler = handlesignal;
        if(sigaction(SIGINT, &act, &old) < 0) {
                oer_debug(DEBUG_FATAL, "main->failed to set SIGINT: %s\n", strerror(errno));
                exit(EXIT_FAILURE);
        }
        if(sigaction(SIGTERM, &act, &old) < 0) {
                oer_debug(DEBUG_FATAL, "main->failed to set SIGTERM: %s\n", strerror(errno));
                exit(EXIT_FAILURE);
        }
#else
        if(signal(SIGINT, handlesignal) == SIG_ERR) {
                oer_debug(DEBUG_FATAL, "main->failed to set SIGINT: %s\n", strerror(errno));
                exit(EXIT_FAILURE);
        }
        if(signal(SIGTERM, handlesignal) == SIG_ERR) {
                oer_debug(DEBUG_FATAL, "main->failed to set SIGTERM: %s\n", strerror(errno));
                exit(EXIT_FAILURE);
        }
#endif
	memset(&mysqldbfrom, 0, sizeof(struct mysqldb));
        if(mysql_init(&mysqldbfrom.mysqldbconn) == NULL) {
                oer_debug(DEBUG_FATAL, "main() : mysql_init() for source failed\n");
                exit(EXIT_FAILURE);
        }
	memset(&mysqldbto, 0, sizeof(struct mysqldb));
        if(mysql_init(&mysqldbto.mysqldbconn) == NULL) {
                oer_debug(DEBUG_FATAL, "main() : mysql_init() for destination failed\n");
                exit(EXIT_FAILURE);
        }
	if(!loadconf(config)) {
                oer_debug(DEBUG_FATAL, "main() : loadconf() failed\n");
		mysql_close(&mysqldbfrom.mysqldbconn);
		mysql_close(&mysqldbto.mysqldbconn);
                exit(EXIT_FAILURE);
        }
	if(!dbconnect(&mysqldbfrom)) {
                exit(EXIT_FAILURE);
        }
	if(!dbconnect(&mysqldbto)) {
		mysql_close(&mysqldbfrom.mysqldbconn);
                exit(EXIT_FAILURE);
        }
	if(!do_console) {
		oer_debug(DEBUG_INFO, "main->oersync calling fork()\n");
		newpid = fork();
		if(newpid < 0) {
			oer_debug(DEBUG_INFO, "main->couldn't fork()\n");
			exit(EXIT_FAILURE);
		}
		if(newpid) {
			/* the parent */
			oer_debug(DEBUG_INFO, "main->fork successful, oersync running in background (child pid is %ld, parent exiting)\n", newpid);
			exit(EXIT_SUCCESS);
		}
		/* necessary for some remote progs */
		close(STDIN_FILENO);
		close(STDOUT_FILENO);
		close(STDERR_FILENO);
		do_console = 0;
	}
	if(do_output) {
		snprintf(stringbuffer, HUGESTRINGLEN, "oersync.log");
                if((logto = fopen(stringbuffer, "a+")) == NULL) {
                        oer_debug(DEBUG_INFO, "main->couldn't open log file %s: %s\n", stringbuffer, strerror(errno));
                        exit(EXIT_FAILURE);
                }
                oer_debug(DEBUG_INFO, "main->output gets logged to %s\n", stringbuffer);
        }
	do_tempoutput = 0;
	loopforever = 1;
	do {
		now = time(NULL);
                oer_debug(DEBUG_INFO, "main->sync begins %s", ctime(&now));
		gettimeofday(&tv1, NULL);
		/* first sync from -> to, only do bisync if enabled
		 and the first sync succeeded */
		if(process() && do_twoway) {
			/* switch from <-> to */
			oer_debug(DEBUG_INFO, "main->swapping source and destination\n");
			memcpy(&mysqldbtemp, &mysqldbfrom, sizeof(struct mysqldb));
			memcpy(&mysqldbfrom, &mysqldbto, sizeof(struct mysqldb));
			memcpy(&mysqldbto, &mysqldbtemp, sizeof(struct mysqldb));
			strncpy(identtemp, identfrom, IDENTLEN);
			strncpy(identfrom, identto, IDENTLEN);
			strncpy(identto, identtemp, IDENTLEN);
			/* then sync to -> from, no need to check return value */
			process();
			/* switch back */
			memcpy(&mysqldbtemp, &mysqldbfrom, sizeof(struct mysqldb));
			memcpy(&mysqldbfrom, &mysqldbto, sizeof(struct mysqldb));
			memcpy(&mysqldbto, &mysqldbtemp, sizeof(struct mysqldb));
			strncpy(identtemp, identfrom, IDENTLEN);
			strncpy(identfrom, identto, IDENTLEN);
			strncpy(identto, identtemp, IDENTLEN);
		}
		gettimeofday(&tv2, NULL);
		if(tv2.tv_sec == tv1.tv_sec) {
			snprintf(stringbuffer, HUGESTRINGLEN, "%ld usecs", tv2.tv_usec - tv1.tv_usec);
		} else {
			snprintf(stringbuffer, HUGESTRINGLEN, "%ld secs and %ld usecs", tv2.tv_sec - tv1.tv_sec, ABS(tv2.tv_usec - tv1.tv_usec));
		}
		oer_debug(DEBUG_INFO, "syncing took %s\n", stringbuffer);
		if(do_singlerun) {
			break;
		}
		tv.tv_sec = interval;
		tv.tv_usec = 0;
		oer_debug(DEBUG_INFO, "sleeping %d seconds\n", interval);
		retval = select(0, NULL, NULL, NULL, &tv);
		if(retval < 0) {
			if(errno != EINTR) {
				oer_debug(DEBUG_FATAL, "main() : select() failed\n");
				loopforever = 0;
			}
		} else if(retval > 0) {
			/* should not happen */
			oer_debug(DEBUG_WARNING, "main() : select() returned something to read (??)\n");
		}
		if(do_output) {
			fflush(logto);
		}
	} while(loopforever);
	if(do_output) {
		fclose(logto);
	}
	mysql_close(&mysqldbfrom.mysqldbconn);
	mysql_close(&mysqldbto.mysqldbconn);
	exit(EXIT_SUCCESS);
}

int loadconf(char *config)
{
        FILE *fp;
        char stringbuffer[HUGESTRINGLEN + 1];

        oer_debug(DEBUG_NOISE, "loadconf(\"%s\")\n", config);
        if((fp = fopen(config, "r")) == NULL) {
                return 0;
        }
        while(fgets(stringbuffer, HUGESTRINGLEN, fp)) {
                striplf(stringbuffer);
                if(emptyline(stringbuffer)) {
                        continue;
                }
                if(iscomment(stringbuffer)) {
                        continue;
                }
                if(!parseconf(stringbuffer)) {
                        oer_debug(DEBUG_WARNING, "loadconf() : non-parseable line: '%s'\n", stringbuffer);
                }
        }
        fclose(fp);
        return 1;
}

int parseconf(char *paramline)
{
        int ppos;
        char p1[STRINGLEN + 1];
        char p2[STRINGLEN + 1];
        char p3[STRINGLEN + 1];
        char p4[STRINGLEN + 1];
        char p5[STRINGLEN + 1];
        char p6[STRINGLEN + 1];

        oer_debug(DEBUG_NOISE, "parseconf(\"%s\")\n", paramline);
        ppos = 0;
        if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p1, STRINGLEN, 1)) < 0) {
                return 0;
        }
        if(!strcasecmp(p1, "syncfrom")) {
		if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p2, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p3, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p4, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p5, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p6, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                strncpy(mysqldbfrom.mysqldbdbhost, p2, HOSTLEN);
                strncpy(mysqldbfrom.mysqldbdbname, p3, TINYSTRINGLEN);
                strncpy(mysqldbfrom.mysqldbdbuser, p4, TINYSTRINGLEN);
                strncpy(mysqldbfrom.mysqldbdbpw, p5, TINYSTRINGLEN);
                strncpy(identfrom, p6, IDENTLEN);
                oer_debug(DEBUG_INFO, "source [host: %s] [database: %s] [user: %s] [ident: %s]\n", mysqldbfrom.mysqldbdbhost, mysqldbfrom.mysqldbdbname, mysqldbfrom.mysqldbdbuser, identfrom);
                return 1;
        }
        if(!strcasecmp(p1, "syncto")) {
		if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p2, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p3, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p4, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p5, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p6, STRINGLEN, 1)) < 0) {
                        return 0;
                }
                strncpy(mysqldbto.mysqldbdbhost, p2, HOSTLEN);
                strncpy(mysqldbto.mysqldbdbname, p3, TINYSTRINGLEN);
                strncpy(mysqldbto.mysqldbdbuser, p4, TINYSTRINGLEN);
                strncpy(mysqldbto.mysqldbdbpw, p5, TINYSTRINGLEN);
                strncpy(identto, p6, IDENTLEN);
                oer_debug(DEBUG_INFO, "destination [host: %s] [database: %s] [user: %s] [ident: %s]\n", mysqldbto.mysqldbdbhost, mysqldbto.mysqldbdbname, mysqldbto.mysqldbdbuser, identto);
                return 1;
        }
        if(!strcasecmp(p1, "sync")) {
		if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p2, STRINGLEN, 1)) < 0) {
                        return 0;
                }
		if(!strcasecmp(p2, "all")) {
			do_admins = 1;
			do_autheds = 1;
			do_hostmasks = 1;
			do_passwords = 1;
			do_timestamps = 1;
			do_users = 1;
		} else if(!strcasecmp(p2, "admins")) {
			do_passwords = 1;
		} else if(!strcasecmp(p2, "admins")) {
			do_admins = 1;
		} else if(!strcasecmp(p2, "autheds")) {
			do_autheds = 1;
		} else if(!strcasecmp(p2, "hostmasks")) {
			do_hostmasks = 1;
		} else if(!strcasecmp(p2, "passwords")) {
			do_passwords = 1;
		} else if(!strcasecmp(p2, "timestamps")) {
			do_timestamps = 1;
		} else if(!strcasecmp(p2, "users")) {
			do_users = 1;
		} else {
			return 0;
		}
		return 1;
	}
        if(!strcasecmp(p1, "interval")) {
		if((ppos = parse(paramline, ppos, DEFAULT_DELIM, p2, STRINGLEN, 1)) < 0) {
                        return 0;
                }
		interval = atoi(p2);
		if(interval >= 10 && interval <= 86400) {
			return 1;
		} else {
			interval = 60;
			return 0;
		}
	}
	return 0;
}

int dbconnect(struct mysqldb *mysqldb)
{
        unsigned int timeout;

        oer_debug(DEBUG_NOISE, "dbconnect()\n");
        timeout = 30;
        oer_debug(DEBUG_INFO, "setting options for %s\n", mysqldb->mysqldbdbhost);
        if(mysql_options(&mysqldb->mysqldbconn, MYSQL_OPT_CONNECT_TIMEOUT, (char *) &timeout)) {
                oer_debug(DEBUG_FATAL, "dbconnect() : mysql_options() failed: %s\n", mysql_error(&mysqldb->mysqldbconn));
                return 0;
        }
        if(mysql_options(&mysqldb->mysqldbconn, MYSQL_OPT_COMPRESS, NULL)) {
                oer_debug(DEBUG_FATAL, "dbconnect() : mysql_options() failed: %s\n", mysql_error(&mysqldb->mysqldbconn));
                return 0;
        }
        oer_debug(DEBUG_INFO, "connecting to %s\n", mysqldb->mysqldbdbhost);
        if(mysql_real_connect(&mysqldb->mysqldbconn, mysqldb->mysqldbdbhost, mysqldb->mysqldbdbuser, mysqldb->mysqldbdbpw, mysqldb->mysqldbdbname, 0, NULL, 0) == NULL) {
                oer_debug(DEBUG_FATAL, "dbconnect() : mysql_real_connect() failed: %s\n", mysql_error(&mysqldb->mysqldbconn));
                return 0;
        }
        return 1;
}

void oer_debug(int level, const char *fmt, ...)
{
        va_list ap;

	if(do_tempoutput && level <= do_debug) {
		va_start(ap, fmt);
                vfprintf(stdout, fmt, ap);
                va_end(ap);
                return;
        }
        if(!do_console && !do_output) {
                return;
        }
        if(level <= do_debug) {
                va_start(ap, fmt);
                if(do_output) {
                        vfprintf(logto, fmt, ap);
                }
                if(do_console) {
                        vfprintf(stdout, fmt, ap);
                }
                va_end(ap);
        }
}

void striplf(char *line)
{
        int i;

        for(i = strlen(line) - 1; isspace((int)line[i]) && i >= 0; i--) {
                line[i] = '\0';
        }
}

int emptyline(char *line)
{
        int length;
        int i;

        for(i = 0, length = strlen(line); i < length; i++) {
                if(!isspace((int)line[i])) {
                        return 0;
                }
        }
        return 1;
}

int iscomment(char *line)
{
        int length;
        int i;

        for(i = 0, length = strlen(line); i < length && isspace((int)line[i]); i++) {
                ;
        }
        if(line[i] == '#') {
                return 1;
        }
        return 0;
}

int parse(char *from, int pos, char *delim, char *to, int to_length, int fixed)
{
        int i;
        int j;
        int k;
        int len;
        int found_delim;

        memset(to, 0, to_length + 1);
        i = pos;
        k = 0;
        j = strlen(delim);
        len = strlen(from);
        found_delim = 0;
        while(i < len && k < to_length) {
                /* loop until delim found, respecting 
                   length constraints */
                if(fixed && !strncasecmp(from + i, delim, j)) {
                        found_delim = 1;
                        break;
                }
                if(!fixed && from[i] == delim[0]) {
                        found_delim = 1;
                        break;
                }
                to[k++] = from[i++];
        }
	if(found_delim) {
                /* advance until next non-delim character */
                if(fixed) {
                        i += j;
                } else {
                        while((from[i] == delim[0]) && (i < len)) {
                                i++;
                        }
                }
        }
        if(k >= to_length) {
                /* insufficient buffer space */
                oer_debug(DEBUG_WARNING, "parse() : source length at least %d, destination length %d\n", k, to_length);
                oer_debug(DEBUG_WARNING, "parse() : destination buffer too small!\n");
                return -1;
        }
        to[k] = '\0';
        return i;
}

int oer_doquery(struct mysqldb *mysqldb, char *function, const char *fmt, ...)
{
        va_list ap;
        struct timeval tv1;
        struct timeval tv2;
        char stringbuffer[HUGESTRINGLEN + 1];

        va_start(ap, fmt);
        vsnprintf(stringbuffer, HUGESTRINGLEN, fmt, ap);
        va_end(ap);
        oer_debug(DEBUG_NOISE, "oer_doquery(\"%s\")\n", stringbuffer);
        gettimeofday(&tv1, NULL);
	if(mysql_query(&mysqldb->mysqldbconn, stringbuffer)) {
		oer_debug(DEBUG_ERROR, "oer_doquery() : mysql_query failed: %s\n", mysql_error(&mysqldb->mysqldbconn));
		return 0;
	}
	gettimeofday(&tv2, NULL);
        if(tv2.tv_sec == tv1.tv_sec) {
                snprintf(stringbuffer, HUGESTRINGLEN, "%ld usecs", tv2.tv_usec - tv1.tv_usec);
        } else {
                snprintf(stringbuffer, HUGESTRINGLEN, "%ld secs and %ld usecs", tv2.tv_sec - tv1.tv_sec, ABS(tv2.tv_usec - tv1.tv_usec));
        }
        oer_debug(DEBUG_NOISE, "oer_doquery() : mysql_query took %s\n", stringbuffer);
        return 1;
}

int process()
{
        char queryfrom[HUGESTRINGLEN + 1];
	MYSQL_RES *result;
        MYSQL_ROW row;
	oer_debug(DEBUG_INFO, "syncing...\n");
	/* first sync admins */
	oer_debug(DEBUG_INFO, "admins\n");
	if(!syncadmins()) {
		return 0;
	}
	/* get list of channels and then sync channel at a time */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT name FROM oer_channels WHERE ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "process", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "process() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		/* sync users for channel */
		oer_debug(DEBUG_INFO, "%s users\n", row[0]);
		if(!syncusers(row[0])) {
			mysql_free_result(result);
			return 0;
		}
	}
	mysql_free_result(result);
	return 1;
}

static void handlesignal(int signal)
{
        oer_debug(DEBUG_NOISE, "handlesignal(%d)\n", signal);
        switch(signal) {
        case SIGINT:
                oer_debug(DEBUG_INFO, "handlesignal->received SIGINT, exiting..\n");
		loopforever = 0;
                break;
        case SIGTERM:
                oer_debug(DEBUG_INFO, "handlesignal->received SIGTERM, exiting..\n");
		loopforever = 0;
                break;
	}
}

int syncadmins()
{
        char queryfrom[HUGESTRINGLEN + 1];
        char queryto[HUGESTRINGLEN + 1];
	MYSQL_RES *result;
        MYSQL_ROW row;
	/* delete admins */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_admins WHERE ident = '%s'", identto);
	if(!do_nothing && do_admins) {
		if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
			return 0;
		}
	}
	/* get admins, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT handle, flags FROM oer_admins WHERE ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncadmins", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncadmins() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_admins VALUES('%s','%s', '%s')", row[0], row[1], identto);
		if(!do_nothing && do_admins) {
			if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete hostmasks */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_hostmasks WHERE type = 1 AND ident = '%s'", identto);
	if(!do_nothing && do_hostmasks) {
		if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
			return 0;
		}
	}
	/* get hostmasks, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT hostmask, channel, handle FROM oer_hostmasks WHERE type = 1 AND ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncadmins", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncadmins() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_hostmasks VALUES('%s','%s',1,'%s','%s')", row[0], row[1], row[2], identto);
		if(!do_nothing && do_hostmasks) {
			if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete autheds */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_autheds WHERE ttype = 'admins' AND ident = '%s'", identto);
	if(!do_nothing && do_autheds) {
		if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
			return 0;
		}
	}
	/* get autheds, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT twhen, thandle, tchannel, thostmask FROM oer_autheds WHERE ttype = 'admins' AND ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncadmins", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncadmins() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_autheds VALUES('admins',%lu,'%s','%s','%s','%s')", atol(row[0]), row[1], row[2], row[3], identto);
		if(!do_nothing && do_autheds) {
			if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete passwords */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_passwords WHERE type = 2 AND ident = '%s'", identto);
	if(!do_nothing && do_passwords) {
		if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
			return 0;
		}
	}
	/* get passwords, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT handle, channel, password FROM oer_passwords WHERE type = 2 AND ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncadmins", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncadmins() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_passwords VALUES('%s','%s',2,'%s','%s')", row[0], row[1], row[2], identto);
		if(!do_nothing && do_passwords) {
			if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete timestamps */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_timestamps WHERE ( ttype = 'admins' OR ttype = 'adminautheds' ) AND ident = '%s'", identto);
	if(!do_nothing && do_timestamps) {
		if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
			return 0;
		}
	}
	/* get timestamps, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT ttype, twhen, tchannel FROM oer_timestamps WHERE ( ttype = 'admins' OR ttype = 'adminautheds' ) AND ident = '%s'", identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncadmins", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncadmins() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_timestamps VALUES('%s',%lu,'%s','%s')", row[0], atol(row[1]), row[2], identto);
		if(!do_nothing && do_timestamps) {
			if(!oer_doquery(&mysqldbto, "syncadmins", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	return 1;
}

int syncusers(char *channel)
{
        char queryfrom[HUGESTRINGLEN + 1];
        char queryto[HUGESTRINGLEN + 1];
	MYSQL_RES *result;
        MYSQL_ROW row;
	/* delete users */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_users WHERE channel = '%s' AND ident = '%s'", channel, identto);
	if(!do_nothing && do_users) {
		if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
			return 0;
		}
	}
	/* get users, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT handle, flags FROM oer_users WHERE channel = '%s' AND ident = '%s'", channel, identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncusers", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncusers() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_users VALUES('%s','%s','%s','%s')", row[0], channel, row[1], identto);
		if(!do_nothing && do_users) {
			if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete hostmasks */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_hostmasks WHERE channel = '%s' AND type = 2 AND ident = '%s'", channel, identto);
	if(!do_nothing && do_hostmasks) {
		if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
			return 0;
		}
	}
	/* get hostmasks, insert to destination */
		snprintf(queryfrom, HUGESTRINGLEN, "SELECT hostmask, handle FROM oer_hostmasks WHERE channel = '%s' AND type = 2 AND ident = '%s'", channel, identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncusers", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncusers() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_hostmasks VALUES('%s','%s',%d,'%s','%s')", row[0], channel, 2, row[1], identto);
		if(!do_nothing && do_hostmasks) {
			if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete autheds */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_autheds WHERE ttype = 'users' AND tchannel = '%s' AND ident = '%s'", channel, identto);
	if(!do_nothing && do_autheds) {
		if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
			return 0;
		}
	}
	/* get autheds, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT twhen, thandle, thostmask FROM oer_autheds WHERE ttype = 'users' AND tchannel = '%s' AND ident = '%s'", channel, identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncusers", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncusers() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_autheds VALUES('users',%lu,'%s','%s','%s','%s')", atol(row[0]), row[1], channel, row[2], identto);
		if(!do_nothing && do_autheds) {
			if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete passwords */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_passwords WHERE channel = '%s' AND type = 1 AND ident = '%s'", channel, identto);
	if(!do_nothing && do_passwords) {
		if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
			return 0;
		}
	}
	/* get passwords, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT handle, password FROM oer_passwords WHERE channel = '%s' AND type = 1 AND ident = '%s'", channel, identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncusers", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncusers() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_passwords VALUES('%s','%s',%d,'%s','%s')", row[0], channel, 1, row[1], identto);
		if(!do_nothing && do_passwords) {
			if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	/* delete timestamps */
	snprintf(queryto, HUGESTRINGLEN, "DELETE FROM oer_timestamps WHERE ( ttype = 'users' OR ttype = 'userautheds' ) AND tchannel = '%s' AND ident = '%s'", channel, identto);
	if(!do_nothing && do_timestamps) {
		if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
			return 0;
		}
	}
	/* get timestamps, insert to destination */
	snprintf(queryfrom, HUGESTRINGLEN, "SELECT ttype, twhen FROM oer_timestamps WHERE ( ttype = 'users' OR ttype = 'userautheds' ) AND tchannel = '%s' AND ident = '%s'", channel, identfrom);
	if(!oer_doquery(&mysqldbfrom, "syncusers", queryfrom)) {
		return 0;
	}
	if((result = mysql_store_result(&mysqldbfrom.mysqldbconn)) == NULL) {
		oer_debug(DEBUG_ERROR, "syncusers() : mysql_store_result() failed: %s\n", mysql_error(&mysqldbfrom.mysqldbconn));
		return 0;
	}
	while((row = mysql_fetch_row(result)) != NULL) {
		snprintf(queryto, HUGESTRINGLEN, "INSERT INTO oer_timestamps VALUES('%s',%lu,'%s','%s')", row[0], atol(row[1]), channel, identto);
		if(!do_nothing && do_timestamps) {
			if(!oer_doquery(&mysqldbto, "syncusers", queryto)) {
				mysql_free_result(result);
				return 0;
			}
		}
	}
	mysql_free_result(result);
	return 1;
}
